<?php

namespace App\Http\Controllers;

use App\Models\Discount;
use App\Models\Domain;
use App\Models\Package;
use App\Models\Payment;
use App\Models\Subscription;
use App\Models\User;
use App\Rules\ValidDiscountCode;
use App\Services\ChipService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SubscriptionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, User $user)
    {

        $validated = $request->validate([
            'name' => 'required',
            'price' => 'required',
            'duration_months' => 'required',
            'state' => 'required',
            'city' => 'required',
            'code' => ['nullable',new ValidDiscountCode],

        ]);

        
        
        $discount = Discount::where('code', $request->code)->first();
        $finalPrice = (float) $validated['price'];
        
        
        if($discount && $discount->isValid()){
            if($discount->type === "fixed"){
                $finalPrice -= $discount->value;
            }
            if($discount->type === "percent"){
                $finalPrice -= ($finalPrice * $discount->value / 100);            
            }
            $finalPrice = max(0, $finalPrice);
            $discount->increment('used_count');
            // $finalPrice = number_format($finalPrice, 2, '.', '');
            
        }

        if($validated['name'] !== 'custom'){
            $package = Package::where('name', $validated['name'])->first();
        }else{
            $package = Package::create([
            'name' => $validated['name'],
            'price' => $validated['price'],
            'duration_months' => $validated['duration_months'],
            'description' => $request->input('description'),
            'create_by' => 'client'
        ]);
        }

        
        Log::info('testing:' . now()->toDateTimeString(), ['duration' => $package->duration_months]);

        $startDate = now();
        $endDate   = $startDate->copy()->addMonths((int)$package->duration_months);
        
        $subscription = Subscription::create([
            'user_id' => $user->id,
            'state' => $validated['state'],
            'city' => $validated['city'],
            'package_id' => $package->id,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'status' => 'in progress',
            'discount_id' => $discount ? $discount->id : null
        ]);
        // Log::info('testing:' . now()->toDateTimeString(), ['package' => $package, 'subscription' => $subscription]);
       
        $payment = Payment::create([
            'attempt_date' => now(),
            'subscription_id' => $subscription->id,
            'status' => 'pending',
            'amount' => $finalPrice,
        ]);

        $chip = new ChipService();
        $chipData = $chip->firstCharge($user->email, $finalPrice * 100, $package->name);

        $payment->update([
            'chip_purchase_id' => $chipData['purchase_id']
        ]);



        Log::info('testing:' . now()->toDateTimeString(), ['final price' => $finalPrice, 'chip data' => $chipData]);

        return response()->json([
            'final_price' => $finalPrice,
            'chip_purchase_id' => $chipData['purchase_id'], // optional
            'checkout_url' => $chipData['checkout_url'], // optional
        ]);
        
    }

    public function storeSubscriptionAdmin(Request $request)
    {

        $validated = $request->validate([
            'name' => 'required|unique:packages',
            'price' => 'required',
            'duration_months' => 'required',

        ]);
        Package::create([
            'name' => $validated['name'],
            'price' => $validated['price'],
            'duration_months' => $validated['duration_months'],
            'create_by' => 'admin'
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show()
    {
        return Package::where('create_by', 'admin')->get();
    }
    public function showById(User $user)
    {
        return Subscription::where('user_id', $user->id)->where('status', 'active')
        ->orWhere('status', 'expired')
        ->with('user')->with('package')->first();
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    public function overrideTerminatedClient(Request $request, Subscription $subscription)
    {
        $sub = Subscription::findOrFail($subscription->id);

        $startDate = Carbon::parse($sub->start_date); 
        $currentEndDate = Carbon::parse($sub->end_date);
        $duration = (int) $sub->package->duration_months;

        if (now()->lessThan($currentEndDate)) {
            $endDate = $currentEndDate->copy()->addMonths($duration);
        } else {
            $startDate = now();
            $endDate   = $startDate->copy()->addMonths($duration);
        }
        $sub->status = $request->subscriptionStatus;
        $sub->start_date = now();
        $sub->end_date = $endDate;
        $sub->late_fee = null;
        $sub->save();

        return response()->json(['message' => 'Reminder updated successfully']);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Package $package)
    {
        $package = Package::find($package->id);

        if (!$package) {
            return response()->json(['message' => 'package not found'], 404);
        }
    
        $package->delete();
    
        return response()->json(['message' => 'package deleted successfully']);
    }

    public function renewalAndExpiredSubscriptionPaymentPage(Subscription $subscription){
        return view('redirect_expired_subscription_payment', ['subscription' => $subscription]);
    }
    public function renewPage(Subscription $subscription){
        return view('renew', ['subscription' => $subscription]);
    }

    public function renewalAndExpiredSubscriptionPayment(Subscription $subscription){
        
        
         Log::info('testing:' . now()->toDateTimeString(), ['subscription' => $subscription->user->email]);
        dd($subscription);
        

        $chip = new ChipService();
        $package = $subscription->package;
        $basePrice = $package->price;
        $lateFeePercent = 0; 
        $lateFeeAmount = 0;

        if($subscription->late_fee > 0) {
            $lateFeePercent = $subscription->late_fee;
            $lateFeeAmount = $basePrice * $lateFeePercent;
        }

        $totalPrice = $basePrice + $lateFeeAmount;
        
        $chip = $chip->renewalAndExpiredSubscriptionCharge($subscription->user->email,$subscription->package->name, $totalPrice);

        Payment::create([
            'chip_purchase_id' => $chip['id'],
            'attempt_date' => now(),
            'subscription_id' => $subscription->id,
            'status' => 'pending',
            'amount' => $totalPrice
        ]);

        if($chip['result']) {
            return  response()->json([
            'checkout_url' => $chip['checkout_url'], // optional
        ]);
        } 
    }
    public function renewalAndExpiredSubscriptionPaymentLaravel(Subscription $subscription){
        

        $chip = new ChipService();
        $package = $subscription->package;
        $basePrice = $package->price;
        $lateFeePercent = 0; 
        $lateFeeAmount = 0;

        if($subscription->late_fee > 0) {
            $lateFeePercent = $subscription->late_fee;
            $lateFeeAmount = $basePrice * $lateFeePercent;
        }

        $totalPrice = $basePrice + $lateFeeAmount;
        
        $chip = $chip->renewalAndExpiredSubscriptionCharge($subscription->user->email,$subscription->package->name, $totalPrice);

        $subscription
        ->update(['renew_attempt' => $subscription->renew_attempt ? $subscription->renew_attempt + 1 : 1]);

        Payment::create([
            'chip_purchase_id' => $chip['id'],
            'attempt_date' => now(),
            'subscription_id' => $subscription->id,
            'status' => 'pending',
            'amount' => $totalPrice
        ]);

        if($chip['result']) {
           return redirect()->away($chip['checkout_url']);
            } else {
                return response()->json(['status' => 'failed'], 500);
            }    
    }
}
