<?php

namespace App\Console\Commands;

use App\Models\Reminder;
use App\Models\Subscription;
use App\Notifications\ExpiredSubscription;
use App\Notifications\CheckSubscriptions as NotificationsCheckSubscriptions;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Notification;


class CheckSubscriptions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'subscription:check';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check subscriptions expiry, send reminders, and update status, renewal';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $today = Carbon::today();
        Subscription::where('status', 'active')
        ->where('end_date', '<', $today)
        ->update(['status' => 'expired']);
        
        $subscriptions = Subscription::where('auto_reminder', 1)
    ->whereNotIn('status', ['pending', 'in progress'])
    ->get();
        
        
        foreach($subscriptions as $sub){
            
            $alreadyRemindedToday = Reminder::whereDate('reminder_date', $today)
            ->where('subscription_id', $sub->id)
            ->exists();
            // $endDate = Carbon::parse()
            $daysLeft = $today->diffInDays($sub->end_date, false);            
            $durationDays = $sub->package->duration_months * 30; // approx
            $firstReminder = floor($durationDays * 0.5);   // 50%
            $secondReminder = floor($durationDays * 0.25); // 25%
            $lastReminder = 7;
            
            // Log::info([
            //     'daysleft' => $daysLeft,
            //     'firstreminder' => $firstReminder,
            //     'secondreminder' => $secondReminder,
            // ]);
            // dd('stop');
            if(in_array($daysLeft, [$firstReminder, $secondReminder, $lastReminder])) {
                if(!$alreadyRemindedToday){
                    Notification::route('mail', $sub->user->email)
                    ->notify(new NotificationsCheckSubscriptions($sub));
                    $sub->reminders()->create([
                        'subscription_id' => $sub->id,
                        'user_id' => $sub->user->id,
                        'reminder_type' => 'email', 
                        'status' => 'sent',
                        'reminder_date' => now()
                    ]);
                }
            }
            
            if($daysLeft == 0){
                Log::info('testing:' . now()->toDateTimeString(), ['days left' => $daysLeft]);
                if(!$alreadyRemindedToday){
                    Notification::route('mail', $sub->user->email)
                    ->notify(new ExpiredSubscription($sub));
                    $sub->reminders()->create([
                        'subscription_id' => $sub->id,
                        'user_id' => $sub->user->id,
                        'reminder_type' => 'email', 
                        'status' => 'sent',
                        'reminder_date' => now()
                    ]);
                }
            }
            
            if ($sub->status === 'expired') {
                $daysLate = abs($daysLeft); // how many days late
                Log::info('testing:' . now()->toDateTimeString(), ['days late' => $daysLate >=1]);
                if ($daysLate >= 1 && $daysLate <= 30) {
                    $sub->late_fee = 0.05; // 5%
                    $this->info("Subscription {$sub->id} late fee = 5%");
                    if(!$alreadyRemindedToday){
                        Notification::route('mail', $sub->user->email)
                        ->notify(new ExpiredSubscription($sub));
                        $sub->reminders()->create([
                            'subscription_id' => $sub->id,
                            'user_id' => $sub->user->id,
                            'reminder_type' => 'email', 
                            'status' => 'sent',
                            'reminder_date' => now()
                        ]);
                    }
                } elseif ($daysLate >= 31 && $daysLate <= 60) {
                    $sub->late_fee = 0.10; // 10%
                    $this->info("Subscription {$sub->id} late fee = 10%");
                } elseif ($daysLate > 60) {
                    $sub->status = 'terminated';
                    $sub->late_fee = null; // no need late fee, account terminated
                    $this->info("Subscription {$sub->id} terminated (late > 60 days).");
                }

                $sub->save();
            }
        }
    }
}
